<?php

require_once __DIR__ . '/../db_connect.php';

class CategoryService {
    private $conn;

    public function __construct($db_connection) {
        $this->conn = $db_connection;
    }

    public function addCategory($name, $description) {
        // Kategori adının benzersizliğini kontrol et
        if ($this->isCategoryNameTaken($name)) {
            return ['status' => 'error', 'message' => "Bu kategori adı zaten mevcut."];
        }

        $stmt = $this->conn->prepare("INSERT INTO categories (name, description) VALUES (?, ?)");
        $stmt->bind_param("ss", $name, $description);

        if ($stmt->execute()) {
            $newId = $stmt->insert_id;
            $stmt->close();
            return ['status' => 'success', 'message' => "Kategori başarıyla eklendi!", 'id' => $newId];
        } else {
            $error = $stmt->error;
            $stmt->close();
            return ['status' => 'error', 'message' => "Kategori eklenirken hata: " . $error];
        }
    }

    public function updateCategory($id, $name, $description) {
        // Kategori adının benzersizliğini kontrol et (kendisi hariç)
        if ($this->isCategoryNameTaken($name, $id)) {
            return ['status' => 'error', 'message' => "Bu kategori adı zaten başka bir kategori tarafından kullanılıyor."];
        }

        $stmt = $this->conn->prepare("UPDATE categories SET name = ?, description = ? WHERE id = ?");
        $stmt->bind_param("ssi", $name, $description, $id);

        if ($stmt->execute()) {
            $affected_rows = $stmt->affected_rows;
            $stmt->close();
            if ($affected_rows > 0) {
                return ['status' => 'success', 'message' => "Kategori başarıyla güncellendi!"];
            } else {
                return ['status' => 'info', 'message' => "Güncellenecek kategori bulunamadı veya veri değişmedi."];
            }
        } else {
            $error = $stmt->error;
            $stmt->close();
            return ['status' => 'error', 'message' => "Kategori güncellenirken hata: " . $error];
        }
    }

    public function deleteCategory($id) {
        // Bu kategoriye bağlı alt kategori olup olmadığını kontrol et
        $stmt_check_subcategories = $this->conn->prepare("SELECT id FROM subcategories WHERE categoryId = ?");
        $stmt_check_subcategories->bind_param("i", $id);
        $stmt_check_subcategories->execute();
        $stmt_check_subcategories->store_result();

        if ($stmt_check_subcategories->num_rows > 0) {
            $stmt_check_subcategories->close();
            return ['status' => 'error', 'message' => "Bu kategoriye bağlı alt kategoriler bulunmaktadır. Lütfen önce alt kategorileri silin."];
        }
        $stmt_check_subcategories->close();

        $stmt = $this->conn->prepare("DELETE FROM categories WHERE id = ?");
        $stmt->bind_param("i", $id);

        if ($stmt->execute()) {
            $affected_rows = $stmt->affected_rows;
            $stmt->close();
            if ($affected_rows > 0) {
                return ['status' => 'success', 'message' => "Kategori başarıyla silindi!"];
            } else {
                return ['status' => 'error', 'message' => "Silinecek kategori bulunamadı."];
            }
        } else {
            $error = $stmt->error;
            $stmt->close();
            return ['status' => 'error', 'message' => "Kategori silinirken hata: " . $error];
        }
    }

    public function getCategoryById($id) {
        $stmt = $this->conn->prepare("SELECT id, name, description FROM categories WHERE id = ?");
        $stmt->bind_param("i", $id);
        $stmt->execute();
        $result = $stmt->get_result();
        $data = $result->fetch_assoc();
        $stmt->close();
        return $data;
    }

    public function getAllCategories() {
        $categories = [];
        $sql = "SELECT id, name, description FROM categories ORDER BY name ASC";
        $result = $this->conn->query($sql);

        if ($result && $result->num_rows > 0) {
            while($row = $result->fetch_assoc()) {
                $categories[] = $row;
            }
        }
        return $categories;
    }

    private function isCategoryNameTaken($name, $excludeId = null) {
        $sql = "SELECT id FROM categories WHERE name = ?";
        $params = [$name];
        $types = "s";

        if ($excludeId !== null) {
            $sql .= " AND id != ?";
            $params[] = $excludeId;
            $types .= "i";
        }

        $stmt = $this->conn->prepare($sql);
        $stmt->bind_param($types, ...$params);
        $stmt->execute();
        $stmt->store_result();
        $is_taken = $stmt->num_rows > 0;
        $stmt->close();
        return $is_taken;
    }
}

?> 